//=============================================================================
// RPGツクールMZ - HideEnemy.js
//-----------------------------------------------------------------------------
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc 敵イベントの表示/非表示を制御します。
 * @author emoriiin979
 *
 * @help HideEnemy.js
 *
 * 敵イベントの表示/非表示を制御するためのコマンドを提供します。
 * 以下のプラグインコマンドをイベント中に実行させます。
 *
 * @command pushEnemyId
 * @text 敵ID格納
 * @desc 敵イベントIDを変数に格納します。
 * 敵に勝利した時に実行されることを想定しています。
 *
 * @command resetEnemyIds
 * @text 敵IDリストのリセット
 * @desc 変数に格納済みの敵IDリストを消去します。
 *
 * @command hideEnemies
 * @text 敵の一括非表示
 * @desc 変数に格納済みの全ての敵イベントを一時消去します。
 * マップの初期イベントで実行されることを想定しています。
 *
 */

(() => {
    "use strict";

    const pluginName = "HideEnemy";

    // 敵ID格納
    PluginManager.registerCommand(pluginName, "pushEnemyId", args => {
        $gameTemp.pushHideEnemyId($gameTemp.getCurrentEventId());
    });

    // 敵IDリストのリセット
    PluginManager.registerCommand(pluginName, "resetEnemyIds", args => {
        $gameTemp.resetHideEnemyIds();
    });

    // 敵の一括非表示
    PluginManager.registerCommand(pluginName, "hideEnemies", args => {
        const hideEnemyIds = $gameTemp.getHideEnemyIds();
        for (let i = 0; i < hideEnemyIds.length; i++) {
            $gameMap.eraseEvent(hideEnemyIds[i]);
        }
    });

    //-------------------------------------------------------------------------
    // Game_Temp
    //
    // 討伐済みの敵オブジェクトIDを格納する変数を用意します。

    const _Game_Temp_initialize = Game_Temp.prototype.initialize;
    Game_Temp.prototype.initialize = function() {
        _Game_Temp_initialize.apply(this, arguments);
        this._currentEventId = 0;
        this._hideEnemyIds = {};
        for (const mapId of $dataMapInfos.filter(x => x).map(x => x["id"])) {
            this._hideEnemyIds[mapId] = [];
        }
    };

    Game_Temp.prototype.getHideEnemyIds = function() {
        return this._hideEnemyIds[$gameMap.mapId()];
    };

    Game_Temp.prototype.pushHideEnemyId = function(enemyId) {
        if (this.getHideEnemyIds().indexOf(enemyId) === -1) {
            this._hideEnemyIds[$gameMap.mapId()].push(enemyId);
        }
    };

    Game_Temp.prototype.resetHideEnemyIds = function() {
        this._hideEnemyIds[$gameMap.mapId()] = [];
    };

    Game_Temp.prototype.getCurrentEventId = function() {
        return this._currentEventId;
    };

    Game_Temp.prototype.setCurrentEventId = function(eventId) {
        if (this.getCurrentEventId() !== eventId) {
            this._currentEventId = eventId;
        }
    };

    const _Game_Interpreter_setup = Game_Interpreter.prototype.setup;
    Game_Interpreter.prototype.setup = function(list, eventId) {
        _Game_Interpreter_setup.call(this, list, eventId);
        $gameTemp.setCurrentEventId(eventId);
    };

    const _Game_Interpreter_executeCommand = Game_Interpreter.prototype.executeCommand;
    Game_Interpreter.prototype.executeCommand = function() {
        $gameTemp.setCurrentEventId(this.eventId());
        return _Game_Interpreter_executeCommand.call(this);
    };
})();
